/*
 * sim800.cpp
 * A library for SeeedStudio seeeduino GPRS shield 
 *
 * Copyright (c) 2013 seeed technology inc.
 * Author        :   lawliet zou
 * Create Time   :   Dec 2013
 * Change Log    :
 *
 * The MIT License (MIT)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 * improved by semesin (http://www.semesin.com/project/)
 */
#include "sim800l.h"

SIM800::SIM800(HardwareSerial *serial, byte resetPin,uint32_t baudRate)
{
	_resetPin = resetPin;
	pinMode(_resetPin,OUTPUT);
	serialSIM800 = serial;
	hwSerial = serial;
	hwSerial->begin(baudRate);
	HardwareSerialMode = true;
};
SIM800::SIM800(SoftwareSerial *serial, byte resetPin,uint32_t baudRate)
{
	_resetPin = resetPin;
	pinMode(_resetPin,OUTPUT);
	serialSIM800 = serial;
	swSerial = serial;
	swSerial->begin(baudRate);
	HardwareSerialMode = false;
};

void SIM800::reset(void)
{
	digitalWrite(_resetPin,LOW);
	delay(500);
	digitalWrite(_resetPin,HIGH);
	while(sendATTest() != 0);                
}

int SIM800::init(uint32_t baudrate)
{
	reset();
	
	char cmd[18];
	char resp[24];
	snprintf(cmd, sizeof(cmd), "AT+IPR=%lu\r", baudrate);
	snprintf(resp, sizeof(resp), "AT+IPR=%lu\r\r\nOK\r\n", baudrate);
	int respond;
	
	if((respond = sendCmdAndWaitForResp("AT+CFUN=1\r", "AT+CFUN=1\r\r\nOK\r\n", DEFAULT_TIMEOUT*3)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CPIN?\r", "AT+CPIN?\r\r\n+CPIN: READY\r\n\r\nOK\r\n", DEFAULT_TIMEOUT*3)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp(cmd, resp, DEFAULT_TIMEOUT*3)) != errorOK)
	{
		return respond;
	}
	if(HardwareSerialMode)
	{
		hwSerial->begin(baudrate);
	}
	else
	{
		swSerial->begin(baudrate);
	}
	
	incomingBuffer = "";
	return errorOK;
}

int SIM800::checkSIMStatus(void)
{
	return sendCmdAndWaitForResp("AT+CPIN?\r","AT+CPIN?\r\r\n+CPIN: READY\r\n",DEFAULT_TIMEOUT*3);
}

int SIM800::readBuffer(char *buffer, int *length, unsigned int timeOut)
{
	int i = 0;
	unsigned long timerStart;
	int sumError=0;
	char error[] = "ERROR";
	int sumOK=0;
	char ok[] = "\r\n\r\nOK";
	
	timerStart = millis();
	while(1) {
		while(serialSIM800->available()) 
		{
			char c = serialSIM800->read();
			DEBUG(c);
			buffer[i++] = c;

			sumError = (c==error[sumError]) ? sumError+1 : 0;
			sumOK = (c==ok[sumOK]) ? sumOK+1 : 0;
			
			if(sumError == sizeof(error)-1)
			{
				return errorResponse;
			}
			if(sumOK == sizeof(ok)-1)
			{
				*length = i;
				return errorOK;
			}
			if(i == *length)
			{
				*length = i;
				return errorNoSpace;
			}
		}
		if(millis() - timerStart > timeOut) 
		{
			*length = i;
			return errorTimeout;
		}
	}
}

void SIM800::sendCmd(const char* cmd)
{
	serialSIM800->write(cmd);
}

int SIM800::sendATTest(void)
{
	return sendCmdAndWaitForResp("AT\r","AT\r\r\nOK\r\n",DEFAULT_TIMEOUT);
}

int SIM800::waitForResp(const char *resp, unsigned int timeout)
{
	int len = strlen(resp);
	int sum=0;
	int sumError=0;
	unsigned long timerStart = millis();
	char error[] = "ERROR";
	
	while(1) {
		while(serialSIM800->available()) {
			char c = serialSIM800->read();
			DEBUG(c);
			
			sum = (c==resp[sum]) ? sum+1 : 0;
			sumError = (c==error[sumError]) ? sumError+1 : 0;
			if(sum == len)
				return errorOK;
			if(sumError == sizeof(error)-1)
				return errorResponse;
		}
		if(millis() - timerStart > timeout) {
				return errorTimeout;
		}
	}
	
}
bool SIM800::waitSIMReady()
{ 
  bool SMSReady = false;
  bool CallReady = false;
	char SMSReadyText[] = "SMS Ready\r\n";
	char CallReadyText[] = "Call Ready\r\n";
	int sumSMSReady=0;
	int sumCallReady=0;
	unsigned long timerStart = millis();
	unsigned long timeout = 30000;

  
	while(1) {
		while(serialSIM800->available()) {
			char c = serialSIM800->read();
			DEBUG(c);
			
			sumSMSReady = (c==SMSReadyText[sumSMSReady]) ? sumSMSReady+1 : 0;
			sumCallReady = (c==CallReadyText[sumCallReady]) ? sumCallReady+1 : 0;

			if(sumSMSReady == sizeof(SMSReadyText)-1)
				SMSReady = true;
			if(sumCallReady == sizeof(CallReadyText)-1)
				CallReady = true;
			if(SMSReady && CallReady)
				return true;
		}
		if(millis() - timerStart > timeout) {
				break;
		}
		if(!((millis() - timerStart) % 2000L)) {
			sendCmd("AT\r");
			delay(500);
		}
	}
	return false;
}

int SIM800::sendCmdAndWaitForResp(const char* cmd, const char *resp, unsigned timeout)
{
	sendCmd(cmd);
	return waitForResp(resp,timeout);
}

void SIM800::serialDebug(void)
{
	while(1) {
		if(serialSIM800->available()){
			Serial.write(serialSIM800->read());
		}
		if(Serial.available()){     
			serialSIM800->write(Serial.read()); 
		}
	}
}
void SIM800::writeDebug(void)
{
	if(Serial.available()){     
		serialSIM800->write(Serial.read()); 
	}
}
int SIM800::join(const char *apn, const char *userName, const char *passWord)
{
	int respond;
	char ipAddr[32];
	char cmd[100];
	char resp[100];
	
	if((respond = sendCmdAndWaitForResp("AT+CIPSHUT\r", "AT+CIPSHUT\r\r\nSHUT OK\r\n", DEFAULT_TIMEOUT*2)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CREG?\r", "AT+CREG?\r\r\n+CREG: 0,1\r\n\r\nOK\r\n", DEFAULT_TIMEOUT)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CIPMUX=0\r", "AT+CIPMUX=0\r\r\nOK\r\n", DEFAULT_TIMEOUT*2)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CIPSTATUS\r", "AT+CIPSTATUS\r\r\nOK\r\n\r\nSTATE: IP INITIAL\r\n", DEFAULT_TIMEOUT*2)) != errorOK)
	{
		return respond;
	}
	snprintf(cmd, sizeof(cmd), "AT+CSTT=\"%s\",\"%s\",\"%s\"\r", apn, userName, passWord);
	snprintf(resp, sizeof(resp), "AT+CSTT=\"%s\",\"%s\",\"%s\"\r\r\nOK\r\n", apn, userName, passWord);
	if((respond = sendCmdAndWaitForResp(cmd, resp, DEFAULT_TIMEOUT*2)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CIPSTATUS\r", "AT+CIPSTATUS\r\r\nOK\r\n\r\nSTATE: IP START\r\n", DEFAULT_TIMEOUT)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CIICR\r", "AT+CIICR\r\r\nOK\r\n", DEFAULT_TIMEOUT)) != errorOK)
	{
		return respond;
	}
	if((respond = sendCmdAndWaitForResp("AT+CIPSTATUS\r", "AT+CIPSTATUS\r\r\nOK\r\n\r\nSTATE: IP GPRSACT\r\n", DEFAULT_TIMEOUT)) != errorOK)
	{
		return respond;
	}
	
	int respondLength = 32;
	sendCmd("AT+CIFSR\r");
	respond = readBuffer(ipAddr,&respondLength, 1000);        
	_ip = str_to_ip(ipAddr+11);
	if(_ip == 0) {
		return respond;
	}
	
	for(byte i =0;i<3;i++)
	{
		if((respond = sendCmdAndWaitForResp("AT+CIPSTATUS\r", "AT+CIPSTATUS\r\r\nOK\r\n\r\nSTATE: IP STATUS\r\n", DEFAULT_TIMEOUT)) == errorOK)
		{
			return errorOK;
		}
	}
	return respond;
}

int SIM800::connectTCP(const char *ip, int port)
{
	char cmd[50];
	char resp[50];
	int respond;

	sprintf(cmd, "AT+CIPSTART=\"TCP\",\"%s\",\"%d\"\r", ip, port);
	sprintf(resp, "AT+CIPSTART=\"TCP\",\"%s\",\"%d\"\r\r\nOK\r\n\r\nCONNECT OK\r\n", ip, port);
	return sendCmdAndWaitForResp(cmd, resp, DEFAULT_TIMEOUT*2);
}

int SIM800::sendTCPData(char *data)
{
	char cmd[32];
	char resp[64];
	int len = strlen(data); 
	int respond;

	snprintf(cmd,sizeof(cmd), "AT+CIPSEND=%d\r",len);
	snprintf(resp,sizeof(resp), "AT+CIPSEND=%d\r\r\n>",len);
	if((respond = sendCmdAndWaitForResp(cmd, resp, DEFAULT_TIMEOUT*2)) != errorOK) {
		return respond;
	}
			
	sendCmd(data);
	serialSIM800->write((char)26);
			
	return waitForResp("\r\nSEND OK\r\n", DEFAULT_TIMEOUT*3);
}

int SIM800::closeTCP(void)
{
	sendCmd("AT+CIPCLOSE\r");
	return errorOK;
}

int SIM800::shutTCP(void)
{
	sendCmd("AT+CIPSHUT\r");
	return errorOK;
}

uint32_t SIM800::str_to_ip(const char* str)
{
	uint32_t ip = 0;
	char *p = (char*)str;
	
	for(int i = 0; i < 4; i++) {
		ip |= atoi(p);
		p = strchr(p, '.');
		if (p == NULL) {
			break;
		}
		if(i < 3) ip <<= 8;
		p++;
	}
	return ip;
}

//HACERR lo de la IP gasta muuuucho espacio (ver .h y todo esto)
char* SIM800::getIPAddress()
{
	uint8_t a = (_ip>>24)&0xff;
	uint8_t b = (_ip>>16)&0xff;
	uint8_t c = (_ip>>8)&0xff;
	uint8_t d = _ip&0xff;

	snprintf(ip_string, sizeof(ip_string), "%d.%d.%d.%d", a,b,c,d);
	return ip_string;
}

int SIM800::networkCheck(void)
{
	int respond;
	delay(1000);
	if((respond = sendCmdAndWaitForResp("AT+CGREG?\r","+CGREG: 0,1",DEFAULT_TIMEOUT*3)) != errorOK) {
		return respond;
	}
	delay(1000);
	if((respond = sendCmdAndWaitForResp("AT+CGATT?\r","+CGATT: 1",DEFAULT_TIMEOUT)) != errorOK) {
		return respond;
	}
	return errorOK;
}

int SIM800::sendSMS(char *number, char *data)
{
	char cmd[32];
	char resp[32];
	int respond;
	
	if((respond = sendCmdAndWaitForResp("AT+CMGF=1\r", "AT+CMGF=1\r\r\nOK\r\n", DEFAULT_TIMEOUT)) != errorOK) {
		return respond;
	}
	delay(500);
	snprintf(cmd, sizeof(cmd), "AT+CMGS=\"%s\"\r", number);
	snprintf(resp, sizeof(resp), "AT+CMGS=\"%s\"\r\r\n> ", number);

	if((respond = sendCmdAndWaitForResp(cmd,resp,DEFAULT_TIMEOUT*10)) == errorOK) 
	{
		serialSIM800->write(data);
		while(waitForResp(data, DEFAULT_TIMEOUT*5) != errorOK);
		serialSIM800->write((char)26);
		if((respond = waitForResp("\r\n\r\nOK\r\n", DEFAULT_TIMEOUT*3)) == errorOK)
		{
			return errorOK;
		}
	}
	return respond;
}

int SIM800::readSMS(int Index, String *messageBuffer, String *numberBuffer)
{
	char cmd[16];// = "AT+CMGL=\"ALL\"\r";
	char *p,*s;
	int respond;
	int respondLength = 300;
	char ResponText[300];
	
	*messageBuffer = "";
	*numberBuffer = "";
	

	if((respond = sendCmdAndWaitForResp("AT+CMGF=1\r","AT+CMGF=1\r\r\nOK\r\n",DEFAULT_TIMEOUT*3)) == errorOK)
	{
		sprintf(cmd,"AT+CMGR=%d\r",Index);
		serialSIM800->write(cmd);
		respond = readBuffer(ResponText, &respondLength, DEFAULT_TIMEOUT*3);
		if(respond == errorOK)
		{
			int i;
			byte csvIndex = 0;
			byte lineIndex = 0;
			
			for(i=0;i<respondLength;i++)
			{
				char c = ResponText[i];
				if(csvIndex == 1)
				{
					if((c != '\"') && (c != ','))
					{
						*numberBuffer += c;
					}
				}
				if(lineIndex == 2)
				{
					if((c != '\"') && (c != '\r') && (c != '\n'))
					{
						*messageBuffer += c;
					}
				}
				if(c == ',')
				{
					csvIndex++;
				}
				else if(c == '\n')
				{
					lineIndex++;
				}
			}			
			return errorOK;
		}
	}
	return respond;
}
void SIM800::SMSListDebug(void)
{
	sendCmd("AT+CMGL=\"ALL\"\r");
	delay(DEFAULT_TIMEOUT);
	while(serialSIM800->available())
	{
		char c = serialSIM800->read();
		DEBUG(c);
	}		
}
byte SIM800::checkEvent()
{
	byte sum = 0;
	char CMTI[] = "+CMTI: \"SM\",";
	bool newIncomingSMS = false;
	String newIncomingSMSIndex = "";
	while(serialSIM800->available())
	{
		char c = serialSIM800->read();
		DEBUG(c);
		if(c == '\n')
		{
			int pos = incomingBuffer.indexOf("+CMTI: \"SM\",");
			if(pos >= 0)
			{
				incomingBuffer = incomingBuffer.substring(pos);
				int index = incomingBuffer.substring(12, incomingBuffer.indexOf('\r')).toInt();
				incomingBuffer = "";
				return index;
			}
			else if(incomingBuffer.indexOf("RING") >= 0)
			{
				incomingBuffer = "";
				return newEventCall;
			}
		}
		else
		{
			incomingBuffer += c;
		}
	}
	return 0;
}

int SIM800::deleteSMS(int index)
{
	char cmd[16];
	char resp[16];
	snprintf(cmd,sizeof(cmd), "AT+CMGD=%d\r",index);
	snprintf(resp,sizeof(resp), "AT+CMGD=%d\r\nOK\r\n",index);
	return sendCmdAndWaitForResp(cmd, resp, DEFAULT_TIMEOUT);
}

int SIM800::callUp(char *number)
{
	char cmd[24];
	int respond;
	
	if((respond = sendCmdAndWaitForResp("AT+COLP=1\r","AT+COLP=1\r\r\nOK\r\n",DEFAULT_TIMEOUT)) != errorOK) {
		return respond;
	}
	delay(1000);
	sprintf(cmd,"\rATD%s;\r", number);
	serialSIM800->write(cmd);
	return errorOK;
}

int SIM800::answer(void)
{
	serialSIM800->write("ATA\r");
	return errorOK;
}

