#include "DMD_Semesin.h"
#ifdef __AVR__
#include "twi_Semesin_Bufferless.h"
#endif

DMDFrame::DMDFrame(byte Width_in_panels, byte Height_in_panels, byte PixelWidthPerPanel, byte PixelHeightPerPanel, bool WaitInterruptOver)
  :
  height(Height_in_panels * PixelHeightPerPanel),
	width_in_panels(Width_in_panels),
	height_in_panels(Height_in_panels),
	pixelWidthPerPanel(PixelWidthPerPanel),
	pixelHeightPerPanel(PixelHeightPerPanel),
	waitInterruptOver(WaitInterruptOver),

  font(0)
{
  width = Width_in_panels * PixelWidthPerPanel;
  row_width_bytes = (width + 7)/8;
  bitmapRowWidthBytes = row_width_bytes * height_in_panels;
  widthBytesToShow = row_width_bytes;
  bitmap = (uint8_t *)malloc(bitmap_bytes());
}

DMDFrame::DMDFrame(uint16_t PixelsWide, byte PixelsHigh, bool WaitInterruptOver)
  :
  height(PixelsHigh),
	width_in_panels(PixelsWide/32),
	height_in_panels(PixelsHigh/16),
	pixelWidthPerPanel(32),
	pixelHeightPerPanel(16),
	waitInterruptOver(WaitInterruptOver),
  font(0)
{
  width = PixelsWide;
  row_width_bytes = (width + 7)/8;
  bitmapRowWidthBytes = row_width_bytes * height_in_panels;
  widthBytesToShow = row_width_bytes;
  bitmap = (uint8_t *)malloc(bitmap_bytes());
}

// DMDFrame::DMDFrame(const DMDFrame &source) :
  // width(source.width),
  // height(source.height),
  // row_width_bytes(source.row_width_bytes),
  // height_in_panels(source.height_in_panels),
  // font(source.font)
// {
  // bitmap = (uint8_t *)malloc(bitmap_bytes());
  // memcpy((void *)bitmap, (void *)source.bitmap, bitmap_bytes());
// }

// DMDFrame::~DMDFrame()
// {
  // free((void *)bitmap);
// }

void DMDFrame::swapBuffers(DMDFrame &other)
{
#ifdef __AVR__
  // AVR can't write pointers atomically, so need to disable interrupts
  char oldSREG = SREG;
  cli();
#endif
  volatile uint8_t *temp = other.bitmap;
  other.bitmap = this->bitmap;
  this->bitmap = temp;
#ifdef __AVR__
  SREG = oldSREG;
#endif
}

void DMDFrame::scanDisplay()
{
}

void DMDFrame::setPeriod(long microseconds)//timer 1
{
#ifdef __AVR__
  uint8_t clockSelectBits;
  TCCR1A = 0;                 // clear control register A 
  TCCR1B = _BV(WGM13);        // set mode 8: phase and frequency correct pwm, stop the timer

  long cycles = (F_CPU / 2000000L) * microseconds;                                // the counter runs backwards after TOP, interrupt is at BOTTOM so divide microseconds by 2
  if(cycles < 0xFFFF)              clockSelectBits = _BV(CS10);              // no prescale, full xtal
  else if((cycles >>= 3) < 0xFFFF) clockSelectBits = _BV(CS11);              // prescale by /8
  else if((cycles >>= 3) < 0xFFFF) clockSelectBits = _BV(CS11) | _BV(CS10);  // prescale by /64
  else if((cycles >>= 2) < 0xFFFF) clockSelectBits = _BV(CS12);              // prescale by /256
  else if((cycles >>= 2) < 0xFFFF) clockSelectBits = _BV(CS12) | _BV(CS10);  // prescale by /1024
  else        cycles = 0xFFFF - 1, clockSelectBits = _BV(CS12) | _BV(CS10);  // request was out of bounds, set as maximum
	
  
  uint8_t oldSREG = SREG;				
  cli();							// Disable interrupts for 16 bit register access
  ICR1 = cycles;                                          // ICR1 is TOP in p & f correct pwm mode
  SREG = oldSREG;
  
  TCCR1B &= ~(_BV(CS10) | _BV(CS11) | _BV(CS12));
  TCCR1B |= clockSelectBits;                                          // reset clock select register, and starts the clock
#elif defined (ESP8266)

#elif STM32_MCU_SERIES == STM32_SERIES_F1
  Timer2.setPeriod(microseconds);

#endif
	
}


const uint8_t pixelBitmask[] = {
  0x80,   //0, bit 7
  0x40,   //1, bit 6
  0x20,   //2. bit 5
  0x10,   //3, bit 4
  0x08,   //4, bit 3
  0x04,   //5, bit 2
  0x02,   //6, bit 1
  0x01    //7, bit 0
};

// Set a single LED on or off. Remember that the pixel array is inverted (bit set = LED off)
void DMDFrame::setPixel(int x, int y, DMDGraphicsMode mode)
{
  if(x >= width || y >= height)
     return;
	 
  if(x < 0 || y < 0)
     return;

  int byte_idx = pixelToBitmapIndex(x,y);
  uint8_t bit = pixelBitmask[x & 0x07];
  
  if(inverse)
  {
	  if(mode == GRAPHICS_ON)
	  {
		  mode = GRAPHICS_OFF;
	  }
	  else if(mode == GRAPHICS_OFF)
	  {
		  mode = GRAPHICS_ON;
	  }
  }
  // uint8_t bit = pixelToBitmask(x);
  
	switch(mode) {
		case GRAPHICS_ON:
			bitmap[byte_idx] &= ~bit; // and with the inverse of the bit - so
			break;
		case GRAPHICS_OFF:
			bitmap[byte_idx] |= bit; // set bit (which turns it off)
			break;
		case GRAPHICS_OR:
			bitmap[byte_idx] = ~(~bitmap[byte_idx] | bit);
			break;
		case GRAPHICS_NOR:
			bitmap[byte_idx] = (~bitmap[byte_idx] | bit);
			break;
		case GRAPHICS_XOR:
			bitmap[byte_idx] ^= bit;
			break;
		case GRAPHICS_INVERSE:
		case GRAPHICS_NOOP:
		break;
	}
}


bool DMDFrame::getPixel(int x, int y)
{
  if(x >= width || y >= height)
	{
     return false;
	}
  int byte_idx = pixelToBitmapIndex(x,y);
  uint8_t bit = pixelBitmask[x & 0x07];
  // uint8_t bit = pixelToBitmask(x);
  bool res = !(bitmap[byte_idx] & bit);
  return res;
}

void DMDFrame::debugPixelLine(unsigned int y, char *buf) { // buf must be large enough (2x pixels+EOL+nul), or we'll overrun
  char *currentPixel = buf;
  for(int x=0;x < width;x++) {
    bool set = getPixel(x,y);
    if(set) {
      *currentPixel='[';
      currentPixel++;
      *currentPixel=']';
    } else {
        *currentPixel='_';
        currentPixel++;
        *currentPixel='_';
    }
    currentPixel++;
  }
  *currentPixel ='\n';
  currentPixel++;
  *currentPixel = 0; // nul terminator
}

void DMDFrame::movePixels(unsigned int from_x, unsigned int from_y,
                         unsigned int to_x, unsigned int to_y,
                         unsigned int width, unsigned int height)
{
  // NB: This implementation is actually a copy-erase so
  // it uses more RAM than a real move implementation would
  // do (however bypasses issues around overlapping regions.)

  if(from_x >= this->width || from_y >= this->height
     || to_x >= this->width || to_y >= this->height)
    return;

  DMDFrame to_move = this->subFrame(from_x, from_y, width, height);
  // this->drawFilledBox(from_x,from_y,from_x+width-1,from_y+height-1,GRAPHICS_OFF);
  this->copyFrame(to_move, to_x, to_y);
}

// Set the entire screen
void DMDFrame::fillScreen(byte warnaIsi)
{
	memset((void *)bitmap, inverse ? 0x00 : 0xFF, bitmap_bytes());
}
void DMDFrame::drawImage(int w, int h, byte *image)
{
	volatile byte *alamatBitmap = bitmap;
	uint16_t ptr = 0;
	for (int j=0;j<h;j++)
	{
		for (int k=0;k<w;k++)
		{
			// *alamatBitmap++ = *image++;
			// Serial.print(j);
			// Serial.print(",");
			// Serial.print(k);
			// Serial.print(",");
			// Serial.println(*(image + ptr + (k/8)),HEX);
			
			if(*(image + ptr + (k/8)) & (1<<(7-(k % 8))))
				setPixel(k,h-j-1,GRAPHICS_OFF);
			else
				setPixel(k,h-j-1,GRAPHICS_ON);
			
		}
		ptr += w/8;
	}
}

void DMDFrame::drawLine(int x1, int y1, int x2, int y2, DMDGraphicsMode mode)
{
  int dy = y2 - y1;
  int dx = x2 - x1;
  int stepx, stepy;

  if (dy < 0) {
    dy = -dy;
    stepy = -1;
  } else {
    stepy = 1;
  }
  if (dx < 0) {
    dx = -dx;
    stepx = -1;
  } else {
    stepx = 1;
  }
  dy = dy * 2;
  dx = dx * 2;

  setPixel(x1, y1, mode);
  if (dx > dy) {
    int fraction = dy - (dx / 2);	// same as 2*dy - dx
    while (x1 != x2) {
      if (fraction >= 0) {
        y1 += stepy;
        fraction -= dx;	// same as fraction -= 2*dx
      }
      x1 += stepx;
      fraction += dy;	// same as fraction -= 2*dy
      setPixel(x1, y1, mode);
    }
  } else {
    int fraction = dx - (dy / 2);
    while (y1 != y2) {
      if (fraction >= 0) {
        x1 += stepx;
        fraction -= dy;
      }
      y1 += stepy;
      fraction += dx;
      setPixel(x1, y1, mode);
    }
  }
}

void DMDFrame::drawCircle(unsigned int xCenter, unsigned int yCenter, int radius, DMDGraphicsMode mode)
{
  // Bresenham's circle drawing algorithm
  int x = -radius;
  int y = 0;
  int error = 2-2*radius;
  while(x < 0) {
    setPixel(xCenter-x, yCenter+y, mode);
    setPixel(xCenter-y, yCenter-x, mode);
    setPixel(xCenter+x, yCenter-y, mode);
    setPixel(xCenter+y, yCenter+x, mode);
    radius = error;
    if (radius <= y) error += ++y*2+1;
    if (radius > x || error > y) error += ++x*2+1;
  }
}

void DMDFrame::drawFilledCircle(unsigned int xCenter, unsigned int yCenter, int radius, DMDGraphicsMode mode)
{
  // Bresenham's circle drawing algorithm
  int x = -radius;
  int y = 0;
  int error = 2-2*radius;
  while(x < 0) {
    drawLine(xCenter-x, yCenter+y, xCenter, yCenter, mode);
    drawLine(xCenter-y, yCenter-x, xCenter, yCenter, mode);
    drawLine(xCenter+x, yCenter-y, xCenter, yCenter, mode);
    drawLine(xCenter+y, yCenter+x, xCenter, yCenter, mode);
    radius = error;
    if (radius <= y) error += ++y*2+1;
    if (radius > x || error > y) error += ++x*2+1;
  }
}

void DMDFrame::drawBox(unsigned int x1, unsigned int y1, unsigned int x2, unsigned int y2, DMDGraphicsMode mode)
{
  drawLine(x1, y1, x2, y1, mode);
  drawLine(x2, y1, x2, y2, mode);
  drawLine(x2, y2, x1, y2, mode);
  drawLine(x1, y2, x1, y1, mode);
}

void DMDFrame::drawFilledBox(unsigned int x1, unsigned int y1, unsigned int x2, unsigned int y2, DMDGraphicsMode mode)
{
  for (unsigned int b = x1; b <= x2; b++) {
    drawLine(b, y1, b, y2, mode);
  }
}

void DMDFrame::scrollY(int scrollBy) {
  if(abs(scrollBy) >= height) { // scrolling over the whole display
    // scrolling will erase everything
    drawFilledBox(0, 0, width-1, height-1, GRAPHICS_OFF);
  }
  else if(scrollBy < 0) { // Scroll up
    movePixels(0, -scrollBy, 0, 0, width, height + scrollBy);
    drawFilledBox(0, height+scrollBy, width, height, GRAPHICS_OFF);
  }
  else if(scrollBy > 0) { // Scroll down
    movePixels(0, 0, 0, scrollBy, width, height - scrollBy);
    drawFilledBox(0, 0, width, scrollBy, GRAPHICS_OFF);
  }
}


void DMDFrame::scrollX(int scrollBy) {
  if(abs(scrollBy) >= width) { // scrolling over the whole display!
    // scrolling will erase everything
    drawFilledBox(0, 0, width-1, height-1, GRAPHICS_OFF);
  }
  else if(scrollBy < 0) { // Scroll left
    movePixels(-scrollBy, 0, 0, 0, width + scrollBy, height);
    drawFilledBox(width+scrollBy, 0, width, height, GRAPHICS_OFF);
  }
  else { // Scroll right
    movePixels(0, 0, scrollBy, 0, width - scrollBy, height);
    drawFilledBox(0, 0, scrollBy, height, GRAPHICS_OFF);
  }
}

void DMDFrame::marqueeScrollX(int scrollBy) {
  // Scrolling is basically the same as normal scrolling, but we save/restore the overlapping
  // area in between to create the marquee effect
  scrollBy = scrollBy % width;

  if(scrollBy < 0)  { // Scroll left
    DMDFrame frame = subFrame(0, 0, -scrollBy, height); // save leftmost
    movePixels(-scrollBy, 0, 0, 0, width+scrollBy, height); // move
    copyFrame(frame, width+scrollBy, 0); // drop back at right edge
  } else { // Scroll right
    DMDFrame frame = subFrame(width-scrollBy, 0, scrollBy, height); // save rightmost
    movePixels(0, 0, scrollBy, 0, width-scrollBy, height); // move
    copyFrame(frame, 0, 0); // drop back at left edge
		// movePixels(0, 0, scrollBy, 0, width, height); // move
  }
}

void DMDFrame::marqueeScrollY(int scrollBy) {
  scrollBy = scrollBy % height;

  if(scrollBy < 0) { // Scroll up
    DMDFrame frame = subFrame(0, 0, width, -scrollBy); // save topmost
    movePixels(0, -scrollBy, 0, 0, width, height+scrollBy); // move
    copyFrame(frame, 0, height+scrollBy); // drop back at bottom edge
  } else { // Scroll down
    DMDFrame frame = subFrame(0, height-scrollBy, width, scrollBy); // save bottommost
    movePixels(0, 0, 0, scrollBy, width, height-scrollBy); // move
    copyFrame(frame, 0, 0); // drop back at top edge
  }
}


DMDFrame DMDFrame::subFrame(unsigned int left, unsigned int top, unsigned int width, unsigned int height)
{
  DMDFrame result(width, height);

  if((left % 8) == 0 && (width % 8) == 0) {
    // Copying from/to byte boundaries, can do simple/efficient copies
    for(unsigned int to_y = 0; to_y < height; to_y++) {
      unsigned int from_y = top + to_y;
      unsigned int from_end = pixelToBitmapIndex(left+width,from_y);
      unsigned int to_byte = result.pixelToBitmapIndex(0,to_y);
      for(unsigned int from_byte = pixelToBitmapIndex(left,from_y); from_byte < from_end; from_byte++) {
        result.bitmap[to_byte++] = this->bitmap[from_byte];
      }
    }
  }
  else {
    // Copying not from a byte boundary. Slow pixel-by-pixel for now.
    for(unsigned int to_y = 0; to_y < height; to_y++) {
      for(unsigned int to_x = 0; to_x < width; to_x++) {
        bool val = this->getPixel(to_x+left,to_y+top);
        result.setPixel(to_x,to_y,val ? GRAPHICS_ON : GRAPHICS_OFF);
      }
    }
  }

  return result;
}

void DMDFrame::copyFrame(DMDFrame &from, unsigned int left, unsigned int top)
{
  if((left % 8) == 0 && (from.width % 8) == 0) {
    // Copying rows on byte boundaries, can do simple/efficient copies
    unsigned int to_bottom = top + from.height;
    if(to_bottom > this->height)
      to_bottom = this->height;
    unsigned int to_right = left + from.width;
    if(to_right > this->width)
      to_right = this->width;
    unsigned int from_y = 0;
    for(unsigned int to_y = top; to_y < to_bottom; to_y++) {
      unsigned int to_end = pixelToBitmapIndex(to_right, to_y);
      unsigned int from_byte = from.pixelToBitmapIndex(0, from_y);
      for(unsigned int to_byte = pixelToBitmapIndex(left,to_y); to_byte < to_end; to_byte++) {
        this->bitmap[to_byte] = from.bitmap[from_byte++];
      }
      from_y++;
    }
  }
  else {
    // Copying not to a byte boundary. Slow pixel-by-pixel for now.
    for(unsigned int from_y = 0; from_y < from.height; from_y++) {
      for(unsigned int from_x = 0; from_x < from.width; from_x++) {
        bool val = from.getPixel(from_x,from_y);
        this->setPixel(from_x + left, from_y + top, val ? GRAPHICS_ON : GRAPHICS_OFF);
      }
    }
  }
}

void DMDFrame::moveFrame(int offsetX, int offsetY)
{
	moveFrame(0, 0, width, height, offsetX, offsetY);
}

void DMDFrame::moveFrame(uint8_t left, uint8_t top, uint8_t lebar, uint8_t tinggi, int offsetX, int offsetY)
{
	if(waitInterruptOver)
	{
		scanningFlag = true;
		while(scanningFlag);
	}
	if((offsetX <= 0) && (offsetY == 0))
	{
		for(int from_y = top; from_y < tinggi + top; from_y++) 
		{
			for(int from_x = left - offsetX; from_x < lebar + left; from_x++) 
			{

				int from_byte_idx = pixelToBitmapIndex(from_x, from_y);
				uint8_t from_bit = pixelBitmask[(from_x)  & 0x07];
				
				bool pix = !(bitmap[from_byte_idx] & from_bit);
				
				int to_byte_idx = pixelToBitmapIndex(from_x + offsetX, from_y);
				uint8_t to_bit = pixelBitmask[(from_x + offsetX)  & 0x07];
				
				if(pix)
				{
					bitmap[to_byte_idx] &= ~to_bit;
				}
				else
				{
					bitmap[to_byte_idx] |= to_bit;
				}
			}
		}
	}
	else if((offsetX == 0) || (offsetY > 0))
	{
		delay(1000);
		for(int from_y = top + tinggi-(1+offsetY); from_y >= 0 + top; from_y -= offsetY) 
		{
			for(int from_x = left; from_x < lebar + left; from_x++) 
			{
				int from_byte_idx = pixelToBitmapIndex(from_x, from_y);
				uint8_t from_bit = pixelBitmask[(from_x)  & 0x07];
				
				bool pix = !(bitmap[from_byte_idx] & from_bit);
				
				int to_byte_idx = pixelToBitmapIndex(from_x, from_y + offsetY);
				uint8_t to_bit = pixelBitmask[(from_x)  & 0x07];
				
				if(pix)
				{
					bitmap[to_byte_idx] &= ~to_bit;
				}
				else
				{
					bitmap[to_byte_idx] |= to_bit;
				}
			}
		}
	}
}

void DMDFrame::copySubFrame(DMDFrame *from, uint8_t left, uint8_t top, uint8_t lebar, uint8_t tinggi, int offsetX, int offsetY)
{
	if(waitInterruptOver)
	{
		scanningFlag = true;
		while(scanningFlag);
	}

	for(uint8_t from_y = top; from_y < top + tinggi; from_y++) 
	{
		for(uint8_t from_x = left; from_x < left + lebar; from_x++) 
		{
			int from_byte_idx = pixelToBitmapIndex(from_x, from_y);
			uint8_t from_bit = pixelBitmask[(from_x)  & 0x07];
			
			int to_byte_idx = pixelToBitmapIndex(from_x + offsetX, from_y + offsetY);
			uint8_t to_bit = pixelBitmask[(from_x + offsetX)  & 0x07];
			
			bitmap[to_byte_idx] &= ~to_bit;
			bitmap[to_byte_idx] |= (from->bitmap[from_byte_idx] & from_bit);
			
		}
	}

}


/* Lookup table for DMD pixel locations, marginally faster than bitshifting */
const PROGMEM uint8_t DMD_Pixel_Lut[] = {
  0x80,   //0, bit 7
  0x40,   //1, bit 6
  0x20,   //2. bit 5
  0x10,   //3, bit 4
  0x08,   //4, bit 3
  0x04,   //5, bit 2
  0x02,   //6, bit 1
  0x01    //7, bit 0
};

int DMDFrame::drawImage(const int x, const int y, const uint8_t image, DMDGraphicsMode mode)
{
}

void readMarqueData(EfekMarque *efekMarque)
{
	if(efekMarque->sumber == sumberRAM)
	{
		efekMarque->_data = *(char*)efekMarque->alamat;
	}
	else if(efekMarque->sumber == sumberFlash)
	{
#ifdef __AVR__
		efekMarque->_data = pgm_read_byte(efekMarque->alamat);
#elif defined (ESP8266)
		efekMarque->_data = pgm_read_byte((void*)efekMarque->alamat);
#endif		
	}	
#ifdef __AVR__
	else if(efekMarque->sumber == sumberI2CEEPROM0x57)
	{
		efekMarque->_data = twiRead(0x57, efekMarque->alamat);
	}
#endif	
}

void DMDFrame::marqueeXMinus(EfekMarque *efekMarque)
{
	uint8_t lebarChar;
	selectFont(efekMarque->font);
	
	if(efekMarque->init)
	{
		readMarqueData(efekMarque);
			
		efekMarque->_posisi = efekMarque->kiri + efekMarque->lebar;
		efekMarque->init = false;
		
		for(uint8_t i=0;i<efekMarque->skip;i++)
		{
			lebarChar = charWidth(efekMarque->_data);
			if ((efekMarque->_posisi + lebarChar) <= (efekMarque->kiri + efekMarque->lebar))
			{
				efekMarque->alamat++;
				readMarqueData(efekMarque);
				efekMarque->_posisi += lebarChar + 1;
			}
			efekMarque->_posisi -= efekMarque->step;
		}
	}
	

	if(efekMarque->_data != 0)
	{
		lebarChar = charWidth(efekMarque->_data);
		if ((efekMarque->_posisi + lebarChar) <= (efekMarque->kiri + efekMarque->lebar))
		{
			efekMarque->alamat++;
			readMarqueData(efekMarque);
			efekMarque->_posisi += lebarChar + 1;
		}
		efekMarque->_posisi -= efekMarque->step;

		moveFrame(efekMarque->kiri, efekMarque->atas, efekMarque->lebar, efekMarque->tinggi, -efekMarque->step, 0);
		drawFilledBox(efekMarque->kiri + efekMarque->lebar - (efekMarque->step), efekMarque->atas, efekMarque->kiri + efekMarque->lebar - 1, efekMarque->atas + efekMarque->tinggi - 1, GRAPHICS_OFF);
		if(efekMarque->_data != 0)
		{
			drawCharWidth(efekMarque->_posisi, efekMarque->atas, efekMarque->_data, (efekMarque->kiri + efekMarque->lebar) - efekMarque->_posisi);
		}
	}
	else if(efekMarque->clear)
	{
	      // Serial.print(F("efekMarque->clear = "));
		efekMarque->clear -= efekMarque->step;
		moveFrame(efekMarque->kiri, efekMarque->atas, efekMarque->lebar, efekMarque->tinggi, -efekMarque->step, 0);
		
		drawFilledBox(efekMarque->kiri + efekMarque->lebar - (efekMarque->step), efekMarque->atas, efekMarque->kiri + efekMarque->lebar - 1, efekMarque->atas + efekMarque->tinggi - 1, GRAPHICS_OFF);
		
		if(!efekMarque->clear)
		{
			efekMarque->mode = nonAktif;
		}
		
	}
}

void DMDFrame::marqueeXPlus(EfekMarque *efekMarque)
{
	if(efekMarque->init)
	{
		efekMarque->_panjang = 0;
		for(;;)
		{
			if(*(char*)efekMarque->alamat == 0)
			{
				break;
			}
			efekMarque->alamat++;
			efekMarque->_panjang++;
		}
		efekMarque->alamat--;
		efekMarque->_data = *(char*)efekMarque->alamat;
		efekMarque->_posisi = efekMarque->kiri - charWidth(efekMarque->_data);
		efekMarque->init = false;
		
		for(uint16_t i=0;i<efekMarque->skip;i++)
		{
			efekMarque->_posisi += efekMarque->step;
			if (efekMarque->_posisi > efekMarque->kiri)
			{
				efekMarque->alamat--;
				if(efekMarque->sumber == sumberRAM)
				{
					efekMarque->_data = *(char*)efekMarque->alamat;
				}
				else if(efekMarque->sumber == sumberFlash)
				{
#ifdef __AVR__
					 efekMarque->_data = pgm_read_byte(efekMarque->alamat);
#elif defined (ESP8266)
					 efekMarque->_data = pgm_read_byte((void*)efekMarque->alamat);
#endif					 
				}
				efekMarque->_posisi -= (charWidth(efekMarque->_data) + 1);
				efekMarque->_panjang -= efekMarque->step;
			}
		}
	}

	efekMarque->_posisi += efekMarque->step;
	if (efekMarque->_posisi > efekMarque->kiri + 1)
	{
		efekMarque->alamat--;
		if(efekMarque->sumber == sumberRAM)
		{
			efekMarque->_data = *(char*)efekMarque->alamat;
		}
		else if(efekMarque->sumber == sumberFlash)
		{
#ifdef __AVR__
			 efekMarque->_data = pgm_read_byte(efekMarque->alamat);
#elif defined (ESP8266)
			 efekMarque->_data = pgm_read_byte((void*)efekMarque->alamat);
#endif					 
		}
		efekMarque->_posisi -= (charWidth(efekMarque->_data) + 1);
		efekMarque->_panjang -=  efekMarque->step;
	}


	if(efekMarque->_panjang == 0)
	{
		efekMarque->mode = nonAktif;
		return;
	}

	drawChar(efekMarque->_posisi -1, efekMarque->atas, efekMarque->_data);
}

void DMDFrame::marqueeYInPlus(EfekMarqueIn *efekMarqueIn, DMDFrame *frame)
{
	efekMarqueIn->posisiInit -= efekMarqueIn->step;

	moveFrame(efekMarqueIn->kiri, efekMarqueIn->atas, efekMarqueIn->lebar, efekMarqueIn->tinggi, 0, efekMarqueIn->step);

	copySubFrame(frame, efekMarqueIn->kiri, efekMarqueIn->posisiInit - efekMarqueIn->atas, efekMarqueIn->lebar, efekMarqueIn->step, 0, -(efekMarqueIn->posisiInit - (efekMarqueIn->atas + (efekMarqueIn->step - 1))));
	if(!efekMarqueIn->posisiInit)
	{
		efekMarqueIn->mode = nonAktif;
	}
}

void DMDFrame::wipeYInPlus(EfekWipe *efekWipe, DMDFrame *frame)
{
	copySubFrame(frame, efekWipe->kiri, efekWipe->posisi, efekWipe->lebar, efekWipe->step);

	efekWipe->posisi += efekWipe->step;
	if(efekWipe->posisi >= efekWipe->selesai)
	{
		efekWipe->mode = nonAktif;
	}
}
void DMDFrame::wipeYOutMinus(EfekWipe *efekWipe)
{
	efekWipe->posisi -= efekWipe->step;
	for(uint16_t i=0;i<efekWipe->step;i++)
	{
		drawLine(efekWipe->kiri, efekWipe->posisi + i, efekWipe->kiri + efekWipe->lebar, efekWipe->posisi + i, GRAPHICS_OFF);
	}
	if(efekWipe->posisi <= efekWipe->mulai)
	{
		efekWipe->mode = nonAktif;
	}
}

void DMDFrame::marqueeClearXMinus(EfekMarqueClear *efekMarqueClear)
{
	moveFrame(efekMarqueClear->kiri, efekMarqueClear->atas, efekMarqueClear->lebar, efekMarqueClear->tinggi, -efekMarqueClear->step, 0);
	// moveFrame(-efekMarqueClear->step, 0);
	if(efekMarqueClear->mulai)
	{
		efekMarqueClear->mulai = false;
		efekMarqueClear->_posisiInit = efekMarqueClear->lebar;
		drawFilledBox(efekMarqueClear->kiri + efekMarqueClear->lebar - efekMarqueClear->step, efekMarqueClear->atas, efekMarqueClear->kiri + efekMarqueClear->lebar - efekMarqueClear->step, efekMarqueClear->atas + efekMarqueClear->tinggi - 1, GRAPHICS_OFF);
	}
	efekMarqueClear->_posisiInit -= efekMarqueClear->step;
	if(!efekMarqueClear->_posisiInit)
	{
		efekMarqueClear->mode = nonAktif;
	}
}
void DMDFrame::marqueeClearYPlus(EfekMarqueClear *efekMarqueClear)
{
	moveFrame(efekMarqueClear->kiri, efekMarqueClear->atas, efekMarqueClear->lebar, efekMarqueClear->tinggi, 0, efekMarqueClear->step);
	// moveFrame(0, efekMarqueClear->step);
	if(efekMarqueClear->mulai)
	{
		efekMarqueClear->mulai = false;
		efekMarqueClear->_posisiInit = efekMarqueClear->atas;
		drawFilledBox(efekMarqueClear->kiri, efekMarqueClear->atas, efekMarqueClear->kiri + efekMarqueClear->lebar - 1, efekMarqueClear->atas + efekMarqueClear->step - 1, GRAPHICS_OFF);
	}
	efekMarqueClear->_posisiInit += efekMarqueClear->step;
	if(efekMarqueClear->_posisiInit == efekMarqueClear->tinggi)
	{
		efekMarqueClear->mode = nonAktif;
	}
}

void DMDFrame::marqueeClearYMinus(EfekMarqueClear *efekMarqueClear)
{
	for(byte i=0;i<efekMarqueClear->step;i++)
	{
		drawLine(efekMarqueClear->kiri, efekMarqueClear->atas + efekMarqueClear->tinggi - (i + 1), efekMarqueClear->kiri + efekMarqueClear->lebar-1, efekMarqueClear->atas + efekMarqueClear->tinggi - (i + 1), GRAPHICS_OFF);
	}
	efekMarqueClear->tinggi -= efekMarqueClear->step;
	if(efekMarqueClear->tinggi == efekMarqueClear->atas)
	{
		efekMarqueClear->mode = nonAktif;
	}
}

void DMDFrame::wipeXOutCenter(EfekWipeCenter *efekWipeCenter)
{
	for(uint16_t i=0;i<efekWipeCenter->step1;i++)
	{
		drawLine(efekWipeCenter->tengah - (efekWipeCenter->posisi * efekWipeCenter->step1) + i, efekWipeCenter->atas, efekWipeCenter->tengah - (efekWipeCenter->posisi * efekWipeCenter->step1) + i, efekWipeCenter->atas + efekWipeCenter->tinggi - 1, GRAPHICS_OFF);
	}
	for(uint16_t i=0;i<efekWipeCenter->step2;i++)
	{
		drawLine(efekWipeCenter->tengah + (efekWipeCenter->posisi * efekWipeCenter->step2) + i, efekWipeCenter->atas, efekWipeCenter->tengah + (efekWipeCenter->posisi * efekWipeCenter->step2) + i, efekWipeCenter->atas + efekWipeCenter->tinggi - 1, GRAPHICS_OFF);
	}
	
	if(efekWipeCenter->posisi > efekWipeCenter->berhenti)
	{
		efekWipeCenter->posisi--;
	}
	else if(efekWipeCenter->posisi < efekWipeCenter->berhenti)
	{
		efekWipeCenter->posisi++;
	}
	else if(efekWipeCenter->posisi == efekWipeCenter->berhenti)
	{
		efekWipeCenter->mode = nonAktif;
	}
}
void DMDFrame::wipeXInCenter(EfekWipeCenter *efekWipeCenter, DMDFrame *frame)
{
	copySubFrame(frame, efekWipeCenter->tengah - (efekWipeCenter->posisi * efekWipeCenter->step1), efekWipeCenter->atas, efekWipeCenter->step1, efekWipeCenter->tinggi);

	copySubFrame(frame, efekWipeCenter->tengah + (efekWipeCenter->posisi * efekWipeCenter->step2), efekWipeCenter->atas, efekWipeCenter->step2, efekWipeCenter->tinggi);

	efekWipeCenter->posisi += efekWipeCenter->step1;
	
	if(efekWipeCenter->posisi == efekWipeCenter->berhenti)
	{
		efekWipeCenter->mode = nonAktif;
	}
}
